unit unit_watch_list;

interface

uses
  graphics, classes, sysutils, controls, comctrls, eltree;

type

TWatchObject = class(TObject)
private
  FID             : integer;
  FVariable       : string;
  FValue          : string;
  FELTree         : TElTree;
  FELTreeItem     : TElTreeItem;
  FSelected       : boolean;
  FParent         : TWatchObject;

  procedure GetFullWatch( var FullWatch : string );
  procedure SetValue ( inValue : string );
  function GetFullVariable : string;
public
  { -------------------------------------------------------------------------- }
  constructor Create( inParent : TWatchObject; inID : integer; inElTree : TElTree; inVariable, inValue : string );
  destructor Destroy; override;
  { -------------------------------------------------------------------------- }

  { -------------------------------------------------------------------------- }
  property Parent : TWatchObject read FParent write FParent;
  property Selected : Boolean read FSelected write FSelected;
  property ElTreeItem : TElTreeItem read FElTreeItem write FElTreeItem;
  property ElTree : TELTree read FElTree write FElTree;
  property FullVariable : string read GetFullVariable;
  property Variable : string read FVariable write FVariable;
  property Value : string read FValue write SetValue;
  property ID : integer read FID write FID;
  { -------------------------------------------------------------------------- }
end;

TWatchList = class(TList)
private
  { -------------------------------------------------------------------------- }
  FELTree         : TElTree;
  { -------------------------------------------------------------------------- }

protected
  { -------------------------------------------------------------------------- }
  function GetWatchObject(Index: Integer): TWatchObject;
  procedure SetWatchObject(Index: Integer; Value: TWatchObject);
  procedure SetCount(Value: Integer);
  function GetNextValidID : integer;
  { -------------------------------------------------------------------------- }
public
  { -------------------------------------------------------------------------- }
  constructor Create; overload;
  destructor Destroy; override;
  procedure Clear; override;
  procedure Delete(index: Integer);
  function AddWatch ( parent : TWatchObject; variable, value : string ) : integer;
  function UpdateWatch ( parent : TWatchObject; variable, value : string ) : integer;
  function FindWatch ( parent : TWatchObject; variable : string ) : integer;
  function FindWatchViaTree ( tree_item : TEltreeItem ) : integer;
  function FindWatchViaID ( ID : integer ) : integer;
  procedure RemoveSelected;
  procedure AddSubWatches( inParent : TWatchObject; sub_items : string );
  { -------------------------------------------------------------------------- }

  { -------------------------------------------------------------------------- }
  property ElTree : TELTree read FElTree write FElTree;
  property watches[Index: Integer]: TWatchObject read GetWatchObject write SetWatchObject; default;
  property Count write SetCount;
  { -------------------------------------------------------------------------- }
end;

implementation

uses unit_editor_utils;

{ create object and initialise image indexes }
constructor TWatchList.Create;
begin
  inherited Create;


end;

{ Clear all the WatchObjects from the list and destroy the list. }
destructor TWatchList.Destroy;
begin
  Clear;
  inherited Destroy;
end;

{ Return an WatchObject from the list. }
function TWatchList.GetWatchObject(Index: Integer): TWatchObject;
begin
  Result := TWatchObject(Items[Index]);
end;

{ Set an WatchObject in the list. Free the old WatchObject. }
procedure TWatchList.SetWatchObject(index: Integer; value: TWatchObject);
begin
  watches[index].free;
  items[index] := pointer(value);
end;

{ Clear the list by deleting all WatchObjects in it. }
procedure TWatchList.Clear;
var
  i : Integer;
begin
  if Assigned( FElTree ) then
  begin
    with FElTree do
    begin
      HeaderSections.Clear;

      with HeaderSections.AddSection do
      begin
        Text := 'Watch';
        AutoSize:=True;
      end;

      with HeaderSections.AddSection do
      begin
        Text := 'Value';
        AutoSize:=True;
      end;

      { make headers visible }
      ShowColumns:=True;
    end;
  end;
  
  for i := count-1 downto 0 do
    Delete ( i );

  inherited clear;
end;

{ Delete an WatchObject from the list, freeing the WatchObject }
procedure TWatchList.Delete(index: Integer);
var
  i : integer;
  new_index : integer;
  ThisObject : TWatchObject;
begin
  ThisObject :=  watches[index];

  for i := Count-1 downto 0 do
    if ( watches[i].Parent = ThisObject ) and
       ( i <> index ) then
      Delete( i );

  new_index := indexof( thisObject );

  if ( new_index <> -1 ) then
  begin
    FEltree.items.Delete( watches[new_index].ElTreeItem );
    watches[new_index].Free;

    inherited delete(new_index);
  end;
end;

{ If the list shrinks, free the WatchObjects that are implicitly deleted. }
procedure TWatchList.SetCount(value: Integer);
begin
  while value < count do
    delete(count-1);

  inherited count := value;
end;

{ find the watch at the specified line number }
function TWatchList.Findwatch( parent : TWatchObject; variable : string ) : integer;
var
  i : integer;
begin
  Result := -1;

  for i:=0 to Count-1 do
    if ( watches[i].variable = variable ) and
       ( watches[i].parent = parent ) then
    begin
      Result := i;
      exit;
    end;
end;

{ find the watch at the specified line number }
function TWatchList.FindwatchViaTree( tree_item : TEltreeItem ) : integer;
var
  i : integer;
begin
  Result := -1;

  for i:=0 to Count-1 do
    if ( watches[i].ElTreeItem = tree_item ) then
    begin
      Result := i;
      exit;
    end;
end;

{ find the watch via its ID}
function TWatchList.FindwatchViaID( ID : integer ) : integer;
var
  i : integer;
begin
  Result := -1;

  for i:=0 to Count-1 do
    if ( watches[i].ID = ID ) then
    begin
      Result := i;
      exit;
    end;
end;

{ add a new watch to the list at specified line number }
function TWatchList.AddWatch ( parent : TWatchObject; variable, value : string ) : integer;
var
  watch_index : integer;
  temp_watch  : TWatchObject;
begin
  watch_index := Findwatch ( parent, variable );

  Result := watch_index;

  if ( watch_index = -1 ) then
  begin
    temp_watch := TWatchObject.Create(parent, GetNextValidID, FElTree, variable, value);
    Add ( temp_watch );

    Result := Count -1 ;
  end;
end;

{ add a new watch to the list at specified line number or update existing }
function TWatchList.UpdateWatch ( parent : TWatchObject; variable, value : string ) : integer;
var
  watch_index : integer;
begin
  watch_index := Findwatch ( parent, variable );

  Result := watch_index;

  if ( watch_index = -1 ) then
  begin
    AddWatch ( parent, variable, value );
    Result := Count -1 ;
  end
  else
    watches[watch_index].value := value;
end;

{ removes the selected watches from the list }
procedure TWatchList.RemoveSelected;
var
  i : integer;
begin
  try
    { tell the tree to not redraw }
    FElTree.items.BeginUpdate;
    { find item index }
    for i := 0 to Count-1 do
    begin
      watches[i].Selected := watches[i].ElTreeItem.Selected ;
    end;

    for i := Count-1 downto 0 do
    begin
      if ( watches[i].Selected ) then
        Delete ( i );
    end;

    FEltree.DeSelectAll;

  finally
    { tell tree we are done }
    FELTree.items.EndUpdate;
  end;
end;

{ looks through list for highest id and returns value + 1 }
function TWatchList.GetNextValidID : integer;
var
  i : integer;
begin
  Result := 0;

  for i := 0 to Count-1 do
  begin
    if ( watches[i].ID > Result ) then
      Result := watches[i].ID;
  end;

  Result := Result + 1;
end;

{ add some sub watches to a watch }
procedure TWatchList.AddSubWatches( inParent : TWatchObject; sub_items : string );
var
  i : integer;
  SubWatchCount : integer;
  watch_name_list : TStringList;
begin
  { tell the tree to not redraw }
  FElTree.items.BeginUpdate;

  try
    SubWatchCount := PacketParamCount(' ',sub_items);

    { add watches to a stringlist }
    watch_name_list := TStringList.Create;
    try
      for i:=0 to ( SubWatchCount div 2 ) - 1 do
        watch_name_list.Add( PacketParamStr(' ', (i*2), sub_items) );

      { delete watches that are not in list }
      for i := Count-1 downto 0 do
        if ( watches[i].Parent = inParent ) and
           ( watch_name_list.indexof(watches[i].Variable) = -1 ) then
             Delete( i );
    finally
      watch_name_list.Free;
    end;

    for i:=inParent.ElTreeItem.Count-1 downto 0 do
      if inParent.ElTreeItem.Item[i].data = nil then
        inParent.ElTreeItem.Item[i].Delete;

    for i:=0 to ( SubWatchCount div 2 ) - 1 do
    begin
      UpdateWatch( inParent, PacketParamStr(' ', (i*2), sub_items), PacketParamStr(' ', (i*2)+1, sub_items));
    end;
  finally
    FElTree.items.EndUpdate;
  end;
end;

{ ============================================================================ }

{ create Watch object and set filename }
constructor TWatchObject.Create( inParent : TWatchObject; inID : integer; inElTree : TElTree; inVariable, inValue : string );
begin
  inherited Create;

  FParent := inParent;
  FID := inID;
  FELTreeItem := nil;
  FELTree := inElTree;

  Variable := inVariable;
  Value := inValue;
end;

{ free Watch object }
destructor TWatchObject.Destroy;
begin
  inherited Destroy;
end;

procedure TWatchObject.SetValue ( inValue : string );
var
  ChildTreeItem : TELtreeItem;
begin
  FValue := inValue;

  if ( FELTreeItem = nil ) then
  begin
    if ( FParent <> nil ) then
      ChildTreeItem := FParent.ElTreeItem
    else
      ChildTreeItem := nil;

    FELTreeItem := FElTree.items.AddChildObject( ChildTreeItem, Variable +'    ', Self);
    FElTree.items.AddChild( FELTreeItem, '');
  end;
  
  with FELTreeItem do
    ColumnText.Text := inValue;
end;

{ recurse back through parents and get full watch name }
procedure TWatchObject.GetFullWatch( var FullWatch : string );
begin
  if ( FParent <> nil ) and ( FParent <> Self ) then
    FParent.GetFullWatch( FullWatch )
  else
    FullWatch := '';

  if ( FullWatch <> '' ) then
    FullWatch := FullWatch + '.';

  FullWatch := FullWatch + FVariable;
end;

function TWatchObject.GetFullVariable : string;
begin
  GetFullWatch ( Result );
end;

end.
