unit unit_projectlist;

interface

uses
  classes, sysutils, controls, unit_editorlist, comctrls, eltree, elheader,
  unit_watch_list, Forms{, VCLUnZip };

type

TFileRemove = procedure ( sender : TObject; index : integer; filename : string ) of object;

TProjectObject = class(TObject)
private
  { -------------------------------------------------------------------------- }
  FFilename                 : string;
  FFileAge                  : integer;
  FRelativeFilename         : string;
  FDSOFileAge               : integer;
  FFileAgeAtCompile         : integer;
  FJustCompiled             : boolean;
  FInZip                    : TProjectObject;
  FBreakpointsGot           : boolean;
  FCachedStream             : TMemoryStream;

  function GetJustFile      : string;
  { -------------------------------------------------------------------------- }

public
  { -------------------------------------------------------------------------- }
  constructor Create( inFilename, inRelativeFileName : string );
  destructor Destroy; override;
  { -------------------------------------------------------------------------- }

  { -------------------------------------------------------------------------- }
  property CachedStream : TMemoryStream read FCachedStream write FCachedStream;
  property InZip : TProjectObject read FInZip write FInZip;
  property Justfile : string read GetJustFile;
  property Filename : string read FFilename write FFilename;
  property RelativeFilename : string read FRelativeFilename write FRelativeFilename;
  property FileAge : integer read FFileAge write FFileAge;
  property FileAgeAtCompile : integer read FFileAgeAtCompile write FFileAgeAtCompile;
  property DSOFileAge : integer read FDSOFileAge write FDSOFileAge;
  property JustCompiled : boolean read FJustCompiled write FJustCompiled;
  property BreakpointsGot : boolean read FBreakpointsGot write FBreakpointsGot;
  { -------------------------------------------------------------------------- }
end;

TProjectList = class(TList)
private
  { -------------------------------------------------------------------------- }
  FEditorList               : TEditorList;
  FElTree                   : TElTree;
  FBasePath                 : string;
  FFileImageIndex           : integer;
  FDebugFileImageIndex      : integer;
  FProjectImageIndex        : integer;
  FProjectName              : string;
  FOnFileRemove             : TFileRemove;
  FProjectFilename          : string;
  FModified                 : boolean;
  FWatchList                : TWatchList;
  FZipImageIndex            : integer;
  FDebugFileList            : TStringList;
  { -------------------------------------------------------------------------- }

protected
  { -------------------------------------------------------------------------- }
  function GetProjectObject(Index: Integer): TProjectObject;
  procedure SetProjectObject(Index: Integer; Value: TProjectObject);
  procedure SetCount(Value: Integer);
  procedure SetProjectName ( value : string );
  procedure SetBasePath( value : string );
  { -------------------------------------------------------------------------- }
public
  { -------------------------------------------------------------------------- }
  constructor Create; overload;
  destructor Destroy; override;
  procedure Clear; override;
  procedure Delete(Index: Integer);
  function HaveFile ( filename : string ) : integer;
  function AddFile( filename : string ) : integer;
  procedure PopulateTreeView;
  function IsProjectFilesSelected : boolean;
  procedure OpenSelected;
  procedure DblClick( Sender: TObject );
  procedure RemoveSelected;
  procedure SaveToFile ( filename : string );
  procedure LoadFromFile ( filename : string );
  function HasRelativeFilename( filename : string ) : integer;
  function OpenRelativeFilename ( filename : string ) : integer;
  function GetContents ( index : integer; FileStrings : TStrings ) : boolean;
  function OpenFileByIndex ( index : integer ) : integer;
  procedure ClearBreakpointsGot;
  procedure UpdateIcons;
  { -------------------------------------------------------------------------- }


  { -------------------------------------------------------------------------- }
  property Modified : boolean read FModified write FModified;
  property ProjectFilename : string read FProjectFilename write FProjectFilename;
  property OnFileRemove : TFileRemove read FOnFileRemove write FOnFileRemove;
  property ProjectName : string read FProjectName write SetProjectName;
  property ProjectImageIndex : integer read FProjectImageIndex write FProjectImageIndex;
  property FileImageIndex : integer read FFileImageIndex write FFileImageIndex;
  property DebugFileImageIndex : integer read FDebugFileImageIndex write FDebugFileImageIndex;
  property ZipImageIndex : integer read FZipImageIndex write FZipImageIndex;
  property BasePath : string read FBasePath write SetBasePath;
  property EditorList : TEditorList read FEditorList write FEditorList;
  property DebugFileList : TStringList read FDebugFileList write FDebugFileList;
  property WatchList : TWatchList read FWatchList write FWatchList;
  property ELTree : TELTree read FElTree write FElTree;
  property files[Index: Integer]: TProjectObject read GetProjectObject write SetProjectObject; default;
  property Count write SetCount;
{ -------------------------------------------------------------------------- }
end;

implementation

{ create object and initialise image indexes }
constructor TProjectList.Create;
begin
  inherited Create;

  FDebugFileImageIndex := -1;
  FFileImageIndex := -1;
  FProjectImageIndex:=-1;
  FModified := False;
  FBasePath:='';
end;

{ Clear all the ProjectObjects from the list and destroy the list. }
destructor TProjectList.Destroy;
begin
  Clear;
  inherited Destroy;
end;

{ Return an ProjectObject from the list. }
function TProjectList.GetProjectObject(Index: Integer): TProjectObject;
begin
  Result := TProjectObject(Items[Index]);
end;

{ Set an ProjectObject in the list. Free the old ProjectObject. }
procedure TProjectList.SetProjectObject(index: Integer; value: TProjectObject);
begin
  files[index].free;
  items[index] := pointer(value);
end;

{ Clear the list by deleting all ProjectObjects in it. }
procedure TProjectList.Clear;
var
  i : Integer;
begin
  for i := count-1 downto 0 do
    Delete ( i );

  ProjectName := '';
  ProjectFileName := '';
  BasePath := '';

  inherited clear;

  Fmodified := false;
end;

{ Delete an ProjectObject from the list, freeing the ProjectObject }
procedure TProjectList.Delete(index: Integer);
var
  tree_item : TElTreeItem;
begin
  FModified := True;

  tree_item := FElTree.items.LookForItem( nil, '', files[index], 0, True, False, False, False, False );
  { delete the node from the tree }
  if ( tree_item <> nil ) then
  begin
    FElTree.items.DeleteItem( tree_item );
  end;

  FEditorList.CloseFile ( files[index].filename );

  files[index].Free;

  inherited delete(index);
end;

{ If the list shrinks, free the ProjectObjects that are implicitly deleted. }
procedure TProjectList.SetCount(value: Integer);
begin
  FModified := True;

  while value < count do
    delete(count-1);

  inherited count := value;
end;

{ given a relative path, do we already have the file ? }
function TProjectList.HaveFile ( filename : string ) : integer;
var
  i : integer;
begin
  { set to fail }
  Result := -1;

  for i := 0 to Count-1 do
  begin
    if ( filename = files[i].filename ) then
    begin
      { found a match }
      Result:=i;
      exit;
    end;
  end;
end;

procedure TProjectList.ClearBreakpointsGot;
var
  i : integer;
begin
  for i:=0 to Count-1 do
  begin
    files[i].BreakPointsGot := False;
  end;
end;

{ check if file exists and add it to the project }
function TProjectList.AddFile( filename : string ) : integer;
var
  file_exists   : boolean;
  new_object    : TProjectObject;
  file_index    : longint;
  {VCLUnzip      : TVCLUnzip;}
  ZipTreeItem   : TElTreeItem;
  zip_object    : TProjectObject;
  i             : integer;
begin
  { set result to fail }
  Result := -1;

  { file exist? }
  file_exists := FileExists ( filename );

  if ( file_exists ) then
  begin
    { now do check to see if we have already got it - do not allow duplicates }
    Result:=HaveFile( filename );
    { any existing file ? }
    if ( result = -1 ) then
    begin
      FModified := True;

      { create new project item }
      new_object := TProjectObject.Create ( filename,  ExtractRelativePath(FBasePath, filename) );
      new_object.FileAge := FileAge( filename );

      if FileExists( filename+'.dso' ) then
      begin
        new_object.DSOFileAge := FileAge( filename+'.dso' );
        new_object.FileAgeAtCompile := new_object.FileAge;
      end;

      { add pointer to list }
      Add( new_object );

      file_index := Count-1;

      { is it a zip file ? }
      if ( lowercase(ExtractFileExt( filename )) = '.vl2' ) then
      begin
        { zip file (vl2) }

        ZipTreeItem := FElTree.items.AddChildObject(FElTree.items.GetFirstNode, files[file_index].justfile, files[file_index]);
        { add the zip file name }
        with ZipTreeItem do
        begin
          ColumnText.Add( ExtractFilePath ( files[file_index].relativefilename ));
          { set image index }
          imageindex := FZipImageIndex;
        end;

        { now add the contents of the zip }
        (*VCLUnzip := TVCLUnzip.Create(Application);
        try
          VCLUnzip.ZipName := filename;
          VCLUnzip.ReadZip;

          for i := 0 to VCLUnzip.Count -1 do
          begin
            if ( ( VCLUnzip.ExternalFileAttributes[i] and faDirectory) <> faDirectory ) and
               ( lowercase(ExtractFileExt(VCLUnzip.FullName[i])) = '.cs' ) then
            begin
              { create new project item }
              zip_object := TProjectObject.Create ( VCLUnzip.FullName[i],  VCLUnzip.FullName[i]);

              zip_object.InZip := new_object;

              { add pointer to list }
              Add( zip_object );

              with FElTree, items.AddChildObject(ZipTreeItem, zip_object.justfile, zip_object) do
              begin
                ColumnText.Add( ExtractFilePath ( zip_object.relativefilename ) );
                if ( DebugFileList.indexof( zip_object.relativefilename ) = -1 ) then
                  { set image index }
                  imageindex := FFileImageIndex
                else
                  imageindex := FDebugFileImageIndex

              end;
            end;
          end;
        finally
          VCLUnzip.Free;
        end;       *)
      end
      else
      begin
        { text file }
        with FElTree, items.AddChildObject(items.GetFirstNode, new_object.justfile, new_object) do
        begin
          ColumnText.Add( ExtractFilePath ( new_object.relativefilename ));
          { set image index }
          if ( DebugFileList.indexof( new_object.relativefilename ) = -1 ) then
            imageindex := FFileImageIndex
          else
            imageindex := FDebugFileImageIndex
        end;
      end;

      { set the result to end of list }
      Result:=file_index;
    end;
  end
end;

{ Scan the project list - as some files might now be debuggable }
procedure TProjectList.UpdateIcons;
var
  i : integer;
  tree_item : TElTreeItem;

begin
  { tell the tree to not redraw }
  FELTree.items.BeginUpdate;
  try
    for i := 0 to count -1 do
    begin
      { vl2 file ? }
      if ( lowercase(ExtractFileExt( files[i].relativefilename )) <> '.vl2' ) then
      begin
        tree_item := FElTree.items.LookForItem( nil, '', files[i], 0, True, False, False, False, False );

        if ( Assigned( tree_item ) ) then
        begin
          { no - so set image index }
          if ( DebugFileList.indexof( lowercase(files[i].relativefilename) ) = -1 ) then
            tree_item.imageindex := FFileImageIndex
          else
            tree_item.imageindex := FDebugFileImageIndex
        end;
      end;
    end;
  finally
     FELTree.items.EndUpdate;
  end;
end;

{ fill the treeview with the current project objects }
procedure TProjectList.PopulateTreeView;
var
  root_node : TElTreeItem;
  i         : integer;

begin
  with FElTree do
  begin
    items.Clear;
    HeaderSections.Clear;

    with HeaderSections.AddSection do
    begin
      Text := 'Filename';
      AutoSize:=True;
    end;

    with HeaderSections.AddSection do
    begin
      Text := 'Path';
      AutoSize:=True;
    end;

    { make headers visible }
    ShowColumns:=True;

    { now add the files }
    root_node := items.Add( nil, FProjectName);
    root_node.imageindex := FProjectImageIndex;
    root_node.ColumnText.Text := FBasePath;

    for i:=0 to Count-1 do
    begin
      { add item }
      with items.AddChildObject(root_node, files[i].justfile, files[i]) do
      begin
        ColumnText.Add( ExtractFilePath ( files[i].RelativeFilename ));
        { vl2 file ? }
        if ( lowercase(ExtractFileExt( files[i].relativefilename )) = '.vl2' ) then
        begin
          { show zip icon }
          imageindex := FZipImageIndex;
        end
        else
        begin
          { show file icon }
          { set image index }
          if ( DebugFileList.indexof( lowercase(files[i].relativefilename) ) = -1 ) then
            imageindex := FFileImageIndex
          else
            imageindex := FDebugFileImageIndex
        end;
      end;
    end;


    SortSection := 0;

    FullExpand;

    OnDblClick := DblClick;
  end;

end;

{ scans through selected FELtree items and sees if any are valid projectobjects }
function TProjectList.IsProjectFilesSelected : boolean;
var
  i : integer;
  project_index : integer;
begin
  Result:=False;
  
  { find item index }
  for i := FELTree.Items.Count-1 downto 0 do
  begin
    if ( FEltree.items[i].Selected ) then
    begin
      project_index := Indexof(FEltree.items[i].Data);

      if ( project_index <> -1 ) then
      begin
        Result:=True;
        break;
      end;
    end;
  end;
end;

{ opens all the valid selected project options }
procedure TProjectList.OpenSelected;
var
  i : integer;
  project_index : integer;
begin
  { find item index }
  for i := FELTree.Items.Count-1 downto 0  do
  begin
    if ( FEltree.items[i].Selected ) then
    begin
      project_index := Indexof(FEltree.items[i].Data);

      if ( project_index <> -1 ) then
        OpenFileByIndex( project_index );

    end;
  end;
end;


{ removes the selected files from the project }
procedure TProjectList.RemoveSelected;
var
  i : integer;
  project_index : integer;
begin
  { tell the tree to not redraw }
  FElTree.items.BeginUpdate;
  try
    { find item index }
    for i := FELTree.Items.Count-1 downto 0 do
    begin
      if ( FEltree.items[i].Selected ) then
      begin
        project_index := Indexof(FEltree.items[i].Data);

        if ( project_index <> -1 ) and ( not Assigned(files[project_index].InZip) ) then
        begin
          FEditorList.CloseFile( files[project_index].filename );

          if ( Assigned(FOnFileRemove) ) then
            FOnFileRemove( files[project_index], project_index, files[project_index].filename );

          Delete( project_index );
        end;
      end;
    end;
  finally
    { tell tree we are done }
    FELTree.items.EndUpdate;
  end;
end;

{ issued when a user double clicks on FEltree }
procedure TProjectList.DblClick( Sender: TObject );
begin
  OpenSelected;
end;

{ Save project to a file }
procedure TProjectList.SaveToFile ( filename : string );
var
  i : integer;
  project_file : TStringList;
  relative_path : string;
  watch_index : integer;
begin
  { create the string list }
  project_file := TStringList.Create;

  try
    project_file.Add(Format('ProjectName=%s',[FProjectName]));
    project_file.Add(Format('BasePath=%s',[FBasePath]));
    project_file.Add(Format('NumberOfFiles=%d',[Count]));
    { go through each file, saving the relative file name }
    for i := 0 to Count-1 do
    begin
      { get relative path }
      relative_path:=ExtractRelativePath(FBasePath, files[i].filename);
      { save file path }
      project_file.Add(Format('file%d=%s',[i, relative_path]));
      { save file age }
      project_file.Add(Format('fileage%d=%d',[i, files[i].fileage]));
      { dso file age }
      project_file.Add(Format('dsofileage%d=%d',[i, files[i].dsofileage]));
      { compiled at file age }
      project_file.Add(Format('fileageatcompile%d=%d',[i, files[i].fileageatcompile]));
    end;

    project_file.Add(Format('NumberOfOpenWindows=%d',[FEditorList.Count]));

    { open windows }
    for i := 0 to FEditorList.Count-1 do
    begin
      { compiled at file age }
      project_file.Add(Format('openfile%d=%s',[i, FEditorList[i].filename]));
      project_file.Add(Format('openfileline%d=%d',[i, FEditorList[i].Editor.TopLine]));
      project_file.Add(Format('zipfile%d=%s',[i, FEditorList[i].ZipFilename]));
    end;


    watch_index:=0;

    { set watches }
    for i := 0 to FWatchList.Count-1 do
    begin
      if ( FWatchList[i].Parent = nil ) then
      begin
        project_file.Add(Format('watch%d=%s',[watch_index, FWatchList[i].Variable]));
        inc(watch_index);
      end;
    end;

    project_file.Add(Format('NumberOfWatches=%d',[watch_index]));

    project_file.SaveToFile ( filename );
  finally
    FModified := False;

    { free the string list }
    project_file.Free;
  end;
end;

{ Load project from a file }
procedure TProjectList.LoadFromFile ( filename : string );
var
  i : integer;
  file_count   : integer;
  project_file : TStringList;
  file_index : integer;
  project_index : integer;
  open_count : integer;
  editor_index : integer;
  watches_count : integer;
  ZipFileName : string;
  parent_index : integer;
begin
  { clear existing project }
  Clear;


  { create the string list }
  project_file := TStringList.Create;

  { tell the tree to not redraw }
  FELTree.items.BeginUpdate;
  try
    project_file.LoadFromFile( filename );
    FProjectName:=project_file.values['ProjectName'];
    FBasePath:=IncludeTrailingBackslash(project_file.values['BasePath']);
    FProjectFilename := filename;
    file_count:=StrToInt(project_file.values['NumberOfFiles']);

    PopulateTreeView;

    { change to base path }
    ChDir( FBasePath );

    { go through each file, loading the basepath + filename }
    for i := 0 to file_count-1 do
    begin
      file_index := AddFile ( ExpandFileName(project_file.Values[Format('file%d',[i])]) );

      if ( file_index <> -1 ) then
      begin
        if ( project_file.Values[Format('fileage%d',[i])] <> '' ) then
          files[file_index].FileAge := StrToInt(project_file.Values[Format('fileage%d',[i])]);
        if ( project_file.Values[Format('dsofileage%d',[i])] <> '' ) then
          files[file_index].DSOFileAge := StrToInt(project_file.Values[Format('dsofileage%d',[i])]);
        if ( project_file.Values[Format('fileageatcompile%d',[i])] <> '' ) then
          files[file_index].FileAgeAtCompile := StrToInt(project_file.Values[Format('fileageatcompile%d',[i])]);
      end;
    end;

    { how many edit windows were open ? }
    open_count:=StrToInt(project_file.values['NumberOfOpenWindows']);

    FEditorList.Clear;
    for i := 0 to open_count-1 do
    begin
      ZipFileName := project_file.Values[Format('zipfile%d',[i])];

      editor_index := -1;

      if ( ZipFileName = '' ) then
      begin
        { open the file }
        editor_index := FEditorList.OpenFile(project_file.Values[Format('openfile%d',[i])], False);
        { jump to the line they were at }
      end
      else
      begin
        project_index := HasRelativeFileName( project_file.Values[Format('openfile%d',[i])] );
        if ( project_index <> -1 ) then
          editor_index := OpenFileByIndex( project_index );
      end;

      if ( editor_index <> -1 ) and ( project_file.Values[Format('openfileline%d',[i])] <> '' )  then
        FEditorList[editor_index].Editor.TopLine:=StrToInt(project_file.Values[Format('openfileline%d',[i])]);
    end;


    { how many watches were set ? }
    watches_count:=StrToInt(project_file.values['NumberOfWatches']);

    FWatchList.Clear;
    for i := 0 to watches_count-1 do
    begin
      { open the file }
      FWatchList.AddWatch(nil, project_file.Values[Format('watch%d',[i])],'');
    end;

  finally
    FELTree.items.EndUpdate;

    FModified := False;

    { free the string list }
    project_file.Free;
  end;
end;

function TProjectList.HasRelativeFilename( filename : string ) : integer;
var
  i : integer;
begin
  { set to fail }
  Result := -1;

  for i := 0 to Count-1 do
  begin
    if ( lowercase(filename) = lowercase(files[i].relativefilename) ) then
    begin
      { found a match }
      Result:=i;
      exit;
    end;
  end;
end;

{ open the file in the project with the matching just filename }
function TProjectList.OpenRelativeFilename ( filename : string ) : integer;
var
  file_index : integer;
begin
  Result := -1;

  { find file }
  file_index := HasRelativeFilename ( filename );

  { found one ? }
  if ( file_index <> -1 ) then
    Result := OpenFileByIndex( file_index );
end;

{ get file at contents at specified index }
function TProjectList.GetContents ( index : integer; FileStrings : TStrings ) : boolean;
{var
  VCLUnzip      : TVCLUnzip;}

begin
  Result := False;

  if ( index >= 0 ) and ( index < Count ) then
  begin
    if ( Assigned(files[index].InZip) ) then
    begin
      { cache file in memory once read once }
      if ( not Assigned( files[index].CachedStream ) ) then
      begin
        { extract zip file }
        (*VCLUnzip := TVCLUnzip.Create(Application);
        try
          VCLUnzip.ZipName := files[index].InZip.FileName;
          VCLUnzip.ReadZip;

          files[index].CachedStream := TMemoryStream.Create;
          if ( VCLUnzip.UnZipToStream( files[index].CachedStream, files[index].filename ) = 0 ) then
          begin
            files[index].CachedStream.Free;
            files[index].CachedStream := nil
          end;
        finally
          VCLUnzip.Free;
        end;*)
      end;

      { check we have it now }
      if ( Assigned( files[index].CachedStream ) ) then
      begin
        files[index].CachedStream.Position := 0;
        FileStrings.Clear;
        FileStrings.LoadFromStream( files[index].CachedStream );

        Result := true;
      end
    end
    else
    begin
      if ( FileExists( files[index].filename ) ) then
      begin
        FileStrings.LoadFromFile( files[index].filename );
        Result := True;
      end;
    end;
  end;
end;

{ open file at specified index }
function TProjectList.OpenFileByIndex ( index : integer ) : integer;
var
  {VCLUnzip      : TVCLUnzip;}
  text_stream   : TMemoryStream;
  unzip_file   : integer;
begin
  Result := -1;

  if ( index >= 0 ) and ( index < Count ) then
  begin
    { this in a zip ? }
    if ( Assigned(files[index].InZip) ) then
    begin
      { was it open ? }
      unzip_file := FEditorList.isOpen( files[index].filename );
      { open it regardless }
      Result := FEditorList.OpenFile( files[index].filename, True );
      { was it already open, or is this a new file }
      if ( unzip_file = -1 )  then
      begin
          { new, get contents from zip }
          FEditorList[Result].ZipFileName := files[index].InZip.FileName;
          GetContents( index, FEditorList[Result].Editor.Lines );
      end;
    end { normal file ? }
    else if ( lowercase(ExtractFileExt( files[index].filename )) <> '.vl2' ) then
      Result := FEditorList.OpenFile( files[index].filename, False );
  end;
end;

procedure TProjectList.SetProjectName ( value : string );
begin
  FModified := True;
  FProjectName := Value;

  if ( FElTree.items.Count > 0 ) then
    FElTree.items[0].Text := value;
end;

procedure TProjectList.SetBasePath( value : string );
begin
  FModified := True;
  FBasePath := value;
end;

{ ============================================================================ }

{ create project object and set filename }
constructor TProjectObject.Create( inFilename, inRelativeFileName : string );
begin
  inherited Create;

  FFilename := inFilename;
  FRelativeFileName := inRelativeFileName;
  FInZip := nil;

  FDSOFileAge := 0;
  FFileAge := 0;
  FFileAgeAtCompile := 0;
  FJustCompiled := False;
  FCachedStream := nil;
end;

{ free project object }
destructor TProjectObject.Destroy;
begin
  if ( Assigned(FCachedStream) ) then
    CachedStream.free;
    
  inherited Destroy;
end;

{ return truncated filename }
function TProjectObject.GetJustFile : string;
begin
  Result:=ExtractFileName( FFilename );
end;



end.
