unit unit_function_list;

interface

uses
  graphics, classes, sysutils, controls, comctrls, eltree, DcjTribes2Syn, forms,
  unit_projectlist;

type
scan_states =

(
ssNothing,
ssFunction );

TFunctionSortTypes = (csFunction, csClass);

TfunctionObject = class(TObject)
private
  FFunctionClassName      : string;
  FFunctionName           : string;
  FFilename               : string;
  FLineNumber             : integer;
public
  { -------------------------------------------------------------------------- }
  constructor Create( inFilename, inFunctionClassName, inFunctionName : string; inLineNumber : integer );
  destructor Destroy; override;
  { -------------------------------------------------------------------------- }

  { -------------------------------------------------------------------------- }
  property FunctionClassName : string read FFunctionClassName write FFunctionClassName;
  property FunctionName : string read FFunctionName write FFunctionName;
  property Filename : string read FFilename write FFilename;
  property LineNumber : integer read FLineNumber write FLineNumber;
  { -------------------------------------------------------------------------- }
end;

TfunctionList = class(TList)
private
  { -------------------------------------------------------------------------- }
  FProjectList    : TProjectList;
  FCurrentSort    : TFunctionSortTypes;
  FListView       : TListView;
  FEnabled        : boolean;
  { -------------------------------------------------------------------------- }

protected
  { -------------------------------------------------------------------------- }
  function GetfunctionObject(Index: Integer): TfunctionObject;
  procedure SetfunctionObject(Index: Integer; Value: TfunctionObject);
  procedure SetCount(Value: Integer);
  procedure SetEnabled(value : boolean);
  { -------------------------------------------------------------------------- }
public
  { -------------------------------------------------------------------------- }
  constructor Create; overload;
  destructor Destroy; override;
  procedure Clear; override;
  procedure RemoveAllItems;
  procedure Delete(index: Integer);
  function Addfunction ( FileName, FunctionClassName, FunctionName : string; LineNumber : integer ) : integer;
  procedure DeleteFilesFunctions( FileName : string ) ;
  procedure DeleteFilesFunctionsByIndex( index : integer) ;
  procedure Scan ( filename : string; temp_string_list : TStrings ) ;
  procedure ScanByIndex ( index : integer ) ;
  function FindClassFunction( FunctionClassName, FunctionName : string ) : integer;
  function FindNextFunction( StartIndex : integer; FunctionName : string ) : integer;
  procedure ScanFileAges ( ForceBuild : boolean );
  procedure SaveToFile( BasePath, filename : string );
  procedure LoadFromFile( filename : string );
  procedure DoSort;
  { -------------------------------------------------------------------------- }

  { -------------------------------------------------------------------------- }
  property Enabled : Boolean read FEnabled write SetEnabled;
  property ListView : TListView read FListView write FListView;
  property CurrentSort : TFunctionSortTypes read FCurrentSort write FCurrentSort;
  property ProjectList : TProjectList read FProjectList write FProjectList;
  property functions[Index: Integer]: TfunctionObject read GetfunctionObject write SetfunctionObject; default;
  property Count write SetCount;
  { -------------------------------------------------------------------------- }
end;

function FunctionSort(Item1, Item2: Pointer): Integer;
function ClassSort(Item1, Item2: Pointer): Integer;

implementation

function FunctionSort(Item1, Item2: Pointer): Integer;
begin
  Result := AnsiCompareText(
            lowercase(TfunctionObject(Item1).FunctionName),
            lowercase(TfunctionObject(Item2).FunctionName));
  if ( Result = 0 ) then
     Result := AnsiCompareText(
            lowercase(TfunctionObject(Item1).FunctionClassName),
            lowercase(TfunctionObject(Item2).FunctionClassName));

end;

function ClassSort(Item1, Item2: Pointer): Integer;
begin
  Result := AnsiCompareText(
            lowercase(TfunctionObject(Item1).FunctionClassName),
            lowercase(TfunctionObject(Item2).FunctionClassName));

  if ( Result = 0 ) then
    Result := AnsiCompareText(
            lowercase(TfunctionObject(Item1).FunctionName),
            lowercase(TfunctionObject(Item2).FunctionName));
end;

{ create object and initialise image indexes }
constructor TfunctionList.Create;
begin
  inherited Create;

  FEnabled := True;
end;

{ Clear all the functionObjects from the list and destroy the list. }
destructor TfunctionList.Destroy;
begin
  Clear;
  inherited Destroy;
end;

{ Return an functionObject from the list. }
function TfunctionList.GetfunctionObject(Index: Integer): TfunctionObject;
begin
  Result := TfunctionObject(Items[Index]);
end;

{ Set an functionObject in the list. Free the old functionObject. }
procedure TfunctionList.SetfunctionObject(index: Integer; value: TfunctionObject);
begin
  functions[index].free;
  items[index] := pointer(value);
end;

{ Clear the list by deleting all functionObjects in it. }
procedure TfunctionList.Clear;
var
  i : Integer;
begin
  FListView.items.BeginUpdate;

  try
    for i := count-1 downto 0 do
      Delete ( i );
  finally
    FListView.items.EndUpdate;
  end;

  inherited clear;
end;

{ Clear the list by deleting all functionObjects but
do not free the object }
procedure TfunctionList.RemoveAllItems;
var
  i : Integer;
begin
  FListView.items.BeginUpdate;

  try
    for i := count-1 downto 0 do
      Remove ( functions[i] );
  finally
    FListView.items.EndUpdate;
  end;

  inherited clear;
end;

{ Delete an functionObject from the list, freeing the functionObject }
procedure TfunctionList.Delete(index: Integer);
begin
  functions[index].Free;

  inherited delete(index);

  FListView.items.Count := Count;
end;

{ If the list shrinks, free the functionObjects that are implicitly deleted. }
procedure TfunctionList.SetCount(value: Integer);
begin
  while value < count do
    delete(count-1);

  FListView.items.Count := value;

  inherited count := value;
end;


{ delete the functions for a specific file }
procedure TfunctionList.DeleteFilesFunctions( FileName : string ) ;
var
  i : integer;
begin
  FListView.items.BeginUpdate;

  try
    for i:=Count-1 downto 0 do
      if ( lowercase(functions[i].FileName) = lowercase(FileName) ) then
      begin
        delete(i);
      end;
  finally
    FListView.items.EndUpdate;
  end;
end;

{ delete the functions for a specific file }
procedure TfunctionList.DeleteFilesFunctionsByIndex( index : integer ) ;
var
  i : integer;
begin
  if ( lowercase(ExtractFileExt( ProjectList.files[index].filename )) = '.vl2' ) then
  begin
    { added a zip so we go through zip contents }
    for i := 0 to ProjectList.Count-1 do
    begin
      if ( Assigned(ProjectList.files[i].InZip) ) and
         ( ProjectList.files[i].InZip = ProjectList.files[index] ) then
      begin
        DeleteFilesFunctions( ProjectList.files[i].RelativeFilename  );
      end;
    end;
  end
  else
  begin
    DeleteFilesFunctions( ProjectList.files[index].RelativeFilename );
  end;
end;

{ find the function at the specified line number }
function TfunctionList.FindClassFunction( FunctionClassName, FunctionName : string ) : integer;
var
  i : integer;
begin
  Result := -1;

  for i:=0 to Count-1 do
    if ( lowercase(functions[i].FunctionName) = lowercase(FunctionName) ) and
       ( lowercase(functions[i].FunctionClassName) = lowercase(FunctionClassName) ) then
    begin
      Result := i;
      exit;
    end;
end;

{ find the function at the specified line number }
function TfunctionList.FindNextFunction( startIndex : integer; FunctionName : string ) : integer;
var
  i : integer;
begin
  Result := -1;

  for i:=startIndex to Count-1 do
    if ( lowercase(functions[i].FunctionName) = lowercase(FunctionName) ) then
    begin
      Result := i;
      exit;
    end;
end;

{ add a new function to the list at specified line number }
function TfunctionList.Addfunction ( FileName, FunctionClassName, FunctionName : string; LineNumber : integer ) : integer;
var
  temp_function  : TfunctionObject;

begin
  Result := -1;

  if ( not FEnabled ) then
    exit;
    
  temp_function := TfunctionObject.Create(FileName, FunctionClassName, FunctionName, LineNumber);
  Add ( temp_function );

  Result := Count -1 ;

  FListView.items.Count := Result;
end;

procedure TfunctionList.ScanByIndex ( index : integer ) ;
var
  temp_string_list : TStringList;
  i : integer;
begin
  if ( not FEnabled ) then
    exit;

  ProjectList.Modified := True;

  if ( lowercase(ExtractFileExt( ProjectList.files[index].filename )) = '.vl2' ) then
  begin
    temp_string_list := TStringList.Create;
    try
      { added a zip so we go through zip contents }
      for i := 0 to ProjectList.Count-1 do
      begin
        if ( Assigned(ProjectList.files[i].InZip) ) and
           ( ProjectList.files[i].InZip = ProjectList.files[index] ) then
        begin
          temp_string_list.Clear;
          ProjectList.GetContents(i, temp_string_list);
          Scan ( ProjectList.files[i].filename, temp_string_list );
        end;
      end;
    finally
      temp_string_list.free;
    end;
  end
  else
  begin
    { normal file }
    ProjectList.files[index].FileAge := FileAge( ProjectList.files[index].filename );

    temp_string_list := TStringList.Create;
    try
      ProjectList.GetContents(index, temp_string_list);
      Scan ( ProjectList.files[index].filename, temp_string_list );
    finally
      temp_string_list.free;
    end;
  end;
end;

procedure TfunctionList.Scan ( filename : string; temp_string_list : TStrings ) ;
var
  fHighlighter : TSynTribes2Syn;
  fSource           : string;
  the_token         : string;
  current_state     : scan_states;
  i : integer;
  current_function  : string;
  current_class     : string;
  project_index : integer;
begin
  fHighlighter := TSynTribes2Syn.Create( Application ); 

  try
    for i := 0 to temp_string_list.Count -1 do
    begin
      fSource := temp_string_list[i];

      fHighlighter.SetLine(fSource, i);

      current_state := ssNothing;

      while not fHighlighter.GetEol do
      begin
        the_token := fHighlighter.GetToken;


        { keyword -------------------------------------------------------------}
        if fHighlighter.GetTokenKind = ord(tkKey) then
        begin
          case current_state of
            { Nothing ------------------}
            ssNothing :
            begin
              { is this a function? }
              if ( lowercase(the_token) = 'function' ) then
              begin
                current_state := ssFunction;
                current_function := '';
                current_class := '';
              end;
            end;
            { --------------------------}
          end;
        end
        else

        { identifier ----------------------------------------------------------}
        if fHighlighter.GetTokenKind = ord(tkIdentifier) then
        begin
          current_function := current_function + the_token;
        end
        else


        { symbol --------------------------------------------------------------}
        if fHighlighter.GetTokenKind = ord(tkSymbol) then
        begin
          { open bracket -----------------}
          if ( the_token = '(' ) then
          begin
            case current_state of
              { Function -----------------}
              ssFunction :
              begin
                AddFunction ( ExtractRelativePath(ProjectList.BasePath, filename), current_class, current_function, i+1 );
                current_state := ssNothing;
              end;
              { --------------------------}
            end;
          end
          else
          { scope ------------------------}
          if ( the_token = '::' ) then
          begin
            { function has class }
            current_class := current_function;
            { reset function }
            current_function := '';
          end;
          { ------------------------------}
        end;


        fHighlighter.Next;
      end;
    end;
  finally
    fHighlighter.free;
  end;
end;

{ check modified date of file, look for changes }
procedure TFunctionList.ScanFileAges( ForceBuild : boolean );
var
  i : integer;

begin
  if ( not FEnabled ) then
    exit;


  FListView.items.BeginUpdate;

  try
    for i:=0 to ProjectList.Count-1 do
    begin
      if ( FileExists( ProjectList.files[i].filename ) ) then
      begin
        if ( ProjectList.files[i].FileAge <> FileAge( ProjectList.files[i].filename ) ) or
           ( ForceBuild ) then
        begin
          { tell the function list to redo functions for this filename }
          DeleteFilesFunctionsByIndex( i );
          ScanByIndex( i );

          ProjectList.files[i].FileAge := FileAge( ProjectList.files[i].filename );
        end;
      end;
    end;
  finally
    FListView.items.EndUpdate;
  end;
end;

procedure TFunctionList.SaveToFile( BasePath, filename : string );
var
  i : integer;
  project_file : TStringList;
  relative_path : string;
begin
  if ( not FEnabled ) then
    exit;


  { create the string list }
  project_file := TStringList.Create;

  try
    project_file.Add(BasePath);
    project_file.Add(IntToStr(Count));
    { go through each file, saving the relative file name }
    for i := 0 to Count-1 do
    begin
      { get relative path }
      relative_path:=ExtractRelativePath(BasePath, functions[i].filename);
      { save file path }
      project_file.Add(relative_path);
      { save function name}
      project_file.Add(functions[i].FunctionName);
      { save class name}
      project_file.Add(functions[i].FunctionClassName);
      { save line number}
      project_file.Add(IntToStr(functions[i].LineNumber));
    end;

    project_file.SaveToFile ( filename );
  finally
    { free the string list }
    project_file.Free;
  end;
end;

procedure TFunctionList.DoSort;
begin
  try
    FListView.items.BeginUpdate;
    
    case FCurrentSort of
      csFunction : Sort(FunctionSort);
      csClass : Sort(ClassSort);
    end;
  finally
    FListView.items.EndUpdate;
  end;
end;

{ Load project from a file }
procedure TFunctionList.LoadFromFile( filename : string );
var
  i : integer;
  file_count   : integer;
  project_file : TStringList;
  BasePath : string;
  FunctionName : string;
  FunctionClassName : string;
  LineNumber : integer;
  line_pos   : integer;
begin
  if ( not FEnabled ) then
    exit;
  
  { clear existing project }
  Clear;

  { create the string list }
  project_file := TStringList.Create;

  try
    FListView.items.BeginUpdate;

    project_file.LoadFromFile( filename );
    BasePath:=project_file[0];
    file_count:=StrToInt(project_file[1]);

    { change to base path }
    ChDir( BasePath );

    line_pos := 2;

    { go through each file, loading the basepath + filename }
    for i := 0 to file_count-1 do
    begin
      Filename := project_file[line_pos];
      inc (line_pos, 1);
      FunctionName :=project_file[line_pos];
      inc (line_pos, 1);
      FunctionClassName :=project_file[line_pos];
      inc (line_pos, 1);
      LineNumber := StrToInt(project_file[line_pos]);
      inc (line_pos, 1);

      AddFunction(FileName, FunctionClassName, FunctionName, LineNumber );
    end;
  finally
    { free the string list }
    project_file.Free;

    FListView.items.EndUpdate;
  end;
end;

procedure TFunctionList.SetEnabled(value : boolean);
begin
  if ( value = false ) then
  begin
    try
      Clear;
    finally
      FEnabled := value;
    end;
  end
  else
  begin
    if ( FEnabled = False ) then
    begin
      FEnabled := True;

      ScanFileAges( True );
    end;
  end;
end;

{ ============================================================================ }

{ create function object and set filename }
constructor TfunctionObject.Create( inFilename, inFunctionClassName, inFunctionName : string; inLineNumber : integer );
begin
  inherited Create;

  FFilename := inFilename;
  FFunctionClassName := inFunctionClassName;
  FFunctionName := inFunctionName;
  FLineNumber := inLineNumber;
end;

{ free function object }
destructor TfunctionObject.Destroy;
begin
  inherited Destroy;
end;



end.
