unit unit_editorlist;

interface

uses
  classes, sysutils, controls, SynEditHighlighter, SynEdit, comctrls, dialogs,
  unit_debugger_list, menus, graphics, SynEditKeyCmds, windows;

type

  TDebugSupportPlugin = class(TSynEditPlugin)
  private
    FDebuggerList             : TDebuggerList;
    FImages                   : TImageList;
    FFileName                 : string;
  protected
    procedure AfterPaint(ACanvas: TCanvas; AClip: TRect;
      FirstLine, LastLine: integer); override;
    procedure LinesInserted(FirstLine, Count: integer); override;
    procedure LinesDeleted(FirstLine, Count: integer); override;
  public
      property FileName : string read FFileName write FFileName;
      property Images : TImageList read FImages write FImages;
      property DebuggerList : TDebuggerList read FDebuggerList write FDebuggerList;
  end;


TReplaceTextEvent = procedure(Sender: TObject; const ASearch, AReplace:
    string; Line, Column: integer; var Action: TSynReplaceAction) of object;


TActiveChange = procedure ( editor_index : integer ) of object;
TFileSave = procedure ( sender : TObject; filename : string ) of object;

TEditorObject = class(TObject)
private
  { -------------------------------------------------------------------------- }
  FZipFileName              : string;
  FFilename                 : string;
  FEditor                   : TSynEdit;
  FTabSheet                 : TTabsheet;
  FPageControl              : TPageControl;
  FHighlighter              : TSynCustomHighlighter;
  FOnReplaceTextEvent       : TReplaceTextEvent;
  Fsavedialog_save_file     : TSaveDialog;
  FImages                   : TImageList;
  FDebuggerList             : TDebuggerList;
  FPopupMenu                : TPopupMenu;
  FFoundLine                : integer;
  FLastX                    : integer;
  FLastY                    : integer;
  FDebuggerSupport          : TDebugSupportPlugin;

  FOnFileSave               : TFileSave;
  { -------------------------------------------------------------------------- }


  { -------------------------------------------------------------------------- }
  procedure SetFilename ( value : string );
  function GetFileName : string;
  procedure SetFoundLine( value : integer );
  { -------------------------------------------------------------------------- }
public
  { -------------------------------------------------------------------------- }
  constructor Create( inPageControl : TPageControl ;
                      inHighlighter : TSynCustomHighlighter;
                      inReplaceTextEvent : TReplaceTextEvent;
                      inSavedialog_save_file : TSaveDialog;
                      inImages : TImagelist;
                      inFileSave : TFileSave;
                      inPopUpMenu : TPopupMenu;
                      inAutoIndent,
                      inInsertMode,
                      inUseTabCharacter,
                      inSmartTabs,
                      inCursorBeyondEOF,
                      inCursorBeyondEOL,
                      inShowScrollHint,
                      inTrimTrailingSpaces,
                      inShowLineNumbers : boolean;
                      inTabStops : integer;
                      inEditorFont : string;
                      inFontSize : integer;
                      inFilename : string;
                      inDebuggerList : TDebuggerList;
                      inReadOnly : boolean;
                      inZipFileName : string);

  procedure SetDefaults (
          inAutoIndent,
          inInsertMode,
          inUseTabCharacter,
          inSmartTabs,
          inCursorBeyondEOF,
          inCursorBeyondEOL,
          inShowScrollHint,
          inTrimTrailingSpaces,
          inShowLineNumbers : boolean;
          inTabStops : integer ;
          inEditorFont : string;
          inFontSize : integer);

  destructor Destroy; override;
  procedure TabSheetOnShow ( Sender : TObject );
  function Save( SaveEvenIfNotModified : boolean ) : boolean;
  function SaveRequest : boolean;
  procedure SaveAs ( inFilename : string );
  procedure GotoLine( line_number : integer );
  procedure GutterClick(Sender: TObject; X, Y, Line: Integer; mark: TSynEditMark);
  procedure SpecialLineColors(Sender: TObject; Line: Integer; var Special: Boolean; var FG, BG: TColor);
  procedure FindGotoLine( line_number : integer );

  { -------------------------------------------------------------------------- }


  { -------------------------------------------------------------------------- }
  property ZipFileName : string read FZipFileName write FZipFileName;
  property DebuggerSupport : TDebugSupportPlugin read FDebuggerSupport write FDebuggerSupport;
  property LastX : integer read FLastX write FLastX;
  property LastY : integer read FLastY write FLastY;
  property FoundLine : integer read FFoundLine write SetFoundLine;
  property PopupMenu : TPopupMenu read FPopupMenu write FPopupMenu;
  property OnFileSave : TFileSave read FOnFileSave write FOnFileSave;
  property DebuggerList : TDebuggerList read FDebuggerList write FDebuggerList;
  property Images : TImageList read FImages write FImages;
  property Highlighter : TSynCustomHighlighter read FHighlighter write FHighlighter;
  property PageControl  : TPageControl read FPageControl write FPageControl;
  property Editor       : TSynEdit read FEditor;
  property Filename     : string read GetFilename write SetFilename;
  { -------------------------------------------------------------------------- }
end;

TEditorList = class(TList)
private
  { -------------------------------------------------------------------------- }
  FCurrentActive            : integer;
  FPageControl              : TPageControl;
  FHighlighter              : TSynCustomHighlighter;
  FOnReplaceTextEvent       : TReplaceTextEvent;
  Fsavedialog_save_file     : TSaveDialog;
  FImages                   : TImageList;
  FOnActiveChange           : TActiveChange;
  FOnFileSave               : TFileSave;
  FPopupMenu                : TPopupMenu;

  FAutoIndent               : boolean;
  FInsertMode               : boolean;
  FUseTabCharacter          : boolean;
  FSmartTabs                : boolean;
  FCursorBeyondEOF          : boolean;
  FCursorBeyondEOL          : boolean;
  FShowScrollHint           : boolean;
  FTrimTrailingSpaces       : boolean;
  FShowLineNumbers          : boolean;
  FTabStops                 : integer;
  FEditorFont               : string;
  FFontSize                 : integer;
  FMouseMove                : TMouseMoveEvent;
  FDebuggerList             : TDebuggerList;
  { -------------------------------------------------------------------------- }

protected
  { -------------------------------------------------------------------------- }
  function GetEditorObject(Index: Integer): TEditorObject;
  procedure SetEditorObject(Index: Integer; Value: TEditorObject);
  procedure SetCount(Value: Integer);
  function GetCurrentActive : integer;
  procedure SetCurrentActive( value  : integer );
  { -------------------------------------------------------------------------- }
public
  { -------------------------------------------------------------------------- }
  constructor Create;
  destructor Destroy; override;
  procedure Clear; override;
  procedure Delete(Index: Integer);
  function Close(index: Integer) : boolean;
  function isOpen ( filename : string ) : integer;
  function OpenFile ( filename : string; read_only : boolean ) : integer;
  procedure CloseFile ( filename : string ) ;
  function NewFile : integer;
  function anyModified : boolean;
  procedure PageControlChange(Sender: TObject);
  procedure SetDefaults;
  { -------------------------------------------------------------------------- }


  { -------------------------------------------------------------------------- }
  property OnMouseMove : TMouseMoveEvent read FMouseMove write FMouseMove;

  property DebuggerList : TDebuggerList read FDebuggerList write FDebuggerList;
  property AutoIndent : boolean read FAutoIndent write FAutoIndent;
  property InsertMode : boolean read FInsertMode write FInsertMode;
  property UseTabCharacter : boolean read FUseTabCharacter write FUseTabCharacter;
  property SmartTabs : boolean read FSmartTabs write FSmartTabs;
  property CursorBeyondEOF : boolean read FCursorBeyondEOF write FCursorBeyondEOF;
  property CursorBeyondEOL : boolean read FCursorBeyondEOL write FCursorBeyondEOL;
  property ShowScrollHint : boolean read FShowScrollHint write FShowScrollHint;
  property TrimTrailingSpaces : boolean read FTrimTrailingSpaces write FTrimTrailingSpaces;
  property ShowLineNumbers : boolean read FShowLineNumbers write FShowLineNumbers;
  property TabStops : integer read FTabStops write FTabStops;
  property EditorFont : string read FEditorFont write FEditorFont;
  property FontSize : integer read FFontSize write FFontSize;
  property PopupMenu : TPopupMenu read FPopupMenu write FPopupMenu;
  property OnFileSave : TFileSave read FOnFileSave write FOnFileSave;
  property OnActiveChange : TActiveChange read FOnActiveChange write FOnActiveChange;
  property Images : TImageList read FImages write FImages;
  property savedialog_save_file : TSaveDialog read Fsavedialog_save_file write Fsavedialog_save_file;
  property OnReplaceText : TReplaceTextEvent read FOnReplaceTextEvent write FOnReplaceTextEvent;
  property CurrentActive : integer read GetCurrentActive write SetCurrentActive;
  property Highlighter : TSynCustomHighlighter read FHighlighter write FHighlighter;
  property PageControl : TPageControl read FPageControl write FPageControl;
  property editors[Index: Integer]: TEditorObject read GetEditorObject write SetEditorObject; default;
  property Count write SetCount;
{ -------------------------------------------------------------------------- }
end;

implementation

uses unit_editor_utils;

constructor TEditorList.Create;
begin
  inherited Create;

  FCurrentActive := -1;
end;

{ Clear all the EditorObjects from the list and destroy the list. }
destructor TEditorList.Destroy;
begin
  Clear;
  inherited Destroy;
end;

{ Return an EditorObject from the list. }
function TEditorList.GetEditorObject(Index: Integer): TEditorObject;
begin
  Result := TEditorObject(Items[Index]);
end;

{ Set an EditorObject in the list. Free the old EditorObject. }
procedure TEditorList.SetEditorObject(index: Integer; value: TEditorObject);
begin
  editors[index].free;
  items[index] := pointer(value);
end;

{ Clear the list by deleting all EditorObjects in it. }
procedure TEditorList.Clear;
var
  i : Integer;
begin
  for i := 0 to count-1 do
    editors[i].free;

  inherited clear;
end;

{ Delete an EditorObject from the list, freeing the EditorObject }
procedure TEditorList.Delete(index: Integer);
begin
  { tell the debugger the editor no long exists }
  FDebuggerList.SetEditor( editors[index].filename, nil );

  editors[index].Free;

  if ( index-1 >= 0 ) and ( index-1 <= Count )then
    CurrentActive := index-1;

  inherited delete(index);
end;

{ Close an Editor from the list, does check for modified }
function TEditorList.Close(index: Integer) : boolean;
begin
  Result := False;

  { not modified }
  if ( not editors[index].editor.modified ) then
  begin
    { just close it }
    Delete ( index );
    Result := True;
  end
  else
  begin
    { ask user whether they want to save this file }
    case MessageDlg(Format('Save changes to %s?',[ExtractFileName(editors[index].filename)]),
              mtConfirmation, [mbYes, mbNo, mbCancel], 0) of
      mrYes :
      begin
          { try and save it }
          if ( editors[index].Save( False ) ) then
            Result := True;
      end;

      mrNo :
      begin
        { abandon changes }
        Delete ( index );

        Result := True;
      end;
    end;
  end;
end;

{ If the list shrinks, free the EditorObjects that are implicitly deleted. }
procedure TEditorList.SetCount(value: Integer);
begin
  while value < count do
    delete(count-1);

  inherited count := value;
end;

{ looks file editor with matching filename }
function TEditorList.isOpen ( filename : string ) : integer;
var
  i : integer;
begin
  Result:=-1;

  { look for match }
  for i:=0 to Count-1 do
  begin
    if ( editors[i].Filename = filename ) then
    begin
      Result:=i;
      exit;
    end;
  end;
end;

{ creates or finds a editor object for passed filename, returns index}
function TEditorList.OpenFile( filename : string; read_only : boolean ) : integer;
var
  temp_editor : TEditorObject;
begin
  Result:=isOpen( filename );

  if ( Result = -1 ) then
  begin
    { create object }
    temp_editor:=TEditorObject.Create( FPageControl, FHighlighter,
                 FOnReplaceTextEvent, Fsavedialog_save_file, FImages,
                 FOnFileSave, FPopupMenu, FAutoIndent,
               FInsertMode,FUseTabCharacter,FSmartTabs,FCursorBeyondEOF,
               FCursorBeyondEOL,
               FShowScrollHint,FTrimTrailingSpaces,FShowLineNumbers,
               FTabStops,FEditorFont, FFontSize,
               filename, FDebuggerList, read_only, '');

    temp_editor.editor.OnMouseMove := FMouseMove;

    Add(temp_editor);

    Result := count-1;
  end;

  { tell the debugger the open editor for this file }
  FDebuggerList.SetEditor( editors[result].filename, editors[result].editor );

  CurrentActive := Result;
end;

{ finds a editor object for passed filename, and closes it}
procedure TEditorList.CloseFile( filename : string ) ;
var
  found_object_index : integer;
begin
  found_object_index:=isOpen( filename );

  if ( found_object_index <> -1 ) then
  begin
    Close( found_object_index );
  end;
end;

{ get the index of the active editor - assumes pagecontrol is in sync}
function TEditorList.GetCurrentActive : integer;
begin
  Result:=FPageControl.ActivePageIndex;
end;

{ set the index of the active editor - assumes pagecontrol is in sync}
procedure TEditorList.SetCurrentActive( value  : integer );
begin
  if ( value <> FCurrentActive ) then
  begin
    FCurrentActive := value;
    FPageControl.ActivePageIndex:=value;
    
    if ( Assigned(FOnActiveChange) ) then
      FOnActiveChange ( value );
  end;
end;

{ creates a new page, with a header of untitled.cs }
function TEditorList.NewFile : integer;
var
  temp_editor : TEditorObject;
begin
  { create object }
  temp_editor:=TEditorObject.Create( FPageControl, FHighlighter,
               FOnReplaceTextEvent, FSavedialog_save_file, FImages,
               FOnFileSave, FPopupMenu, FAutoIndent,
               FInsertMode,FUseTabCharacter,FSmartTabs,FCursorBeyondEOF,
               FCursorBeyondEOL,
               FShowScrollHint,FTrimTrailingSpaces,FShowLineNumbers,FTabStops,
               FEditorFont, FFontSize, '', FDebuggerList, False, '');

  temp_editor.editor.OnMouseMove := FMouseMove;

  { add to list }
  Add(temp_editor);

  Result := count-1;
  CurrentActive := Result;
end;

{ goes through list and checks if any editors have been modified }
function TEditorList.anyModified : boolean;
var
  i : integer;
begin
  Result := False;

  for i:=0 to Count-1 do
    if ( editors[i].editor.modified ) then
    begin
      Result:=True;
      break;
    end;
end;

procedure TEditorList.PageControlChange(Sender: TObject);
begin
  CurrentActive := CurrentActive;
end;

procedure TEditorList.SetDefaults;
var
  i : integer;
begin

  for i := 0 to Count -1 do
    editors[i].SetDefaults(
              FAutoIndent,
              FInsertMode,
              FUseTabCharacter,
              FSmartTabs,
              FCursorBeyondEOF,
              FCursorBeyondEOL,
              FShowScrollHint,
              FTrimTrailingSpaces,
              FShowLineNumbers,
              FTabStops,
              FEditorFont,
              FFontSize);
end;



{ ============================================================================ }

constructor TEditorObject.Create( inPageControl : TPageControl ;
                                  inHighlighter : TSynCustomHighlighter;
                                  inReplaceTextEvent : TReplaceTextEvent;
                                  inSavedialog_save_file : TSaveDialog;
                                  inImages : TImagelist;
                                  inFileSave : TFileSave;
                                  inPopUpMenu : TPopupMenu;
                                  inAutoIndent,
                                  inInsertMode,
                                  inUseTabCharacter,
                                  inSmartTabs,
                                  inCursorBeyondEOF,
                                  inCursorBeyondEOL,
                                  inShowScrollHint,
                                  inTrimTrailingSpaces,
                                  inShowLineNumbers : boolean;
                                  inTabStops : integer;
                                  inEditorFont : string;
                                  inFontSize : integer;
                                  inFilename : string;
                                  inDebuggerList : TDebuggerList;
                                  inReadOnly : boolean;
                                  inZipFileName : string);
begin
  inherited Create;
  { link this to correct page control }
  FPageControl:=inPageControl;
  FFileName:=inFilename;
  FZipFileName:=inZipFileName;
  FHighlighter:=inHighlighter;
  FOnReplaceTextEvent := inReplaceTextEvent;
  FTabSheet:=TTabSheet.Create(FPageControl);
  FSavedialog_save_file := inSavedialog_save_file;
  FImages := inImages;
  FOnFileSave := inFileSave;
  FPopUpMenu := inPopUpMenu;
  FDebuggerList := inDebuggerList;

  FFoundLine := -1;

  { create a new sheet }
  with FTabSheet do
  begin
    PageControl := FPageControl;
    Caption := ExtractFileName(FileName);
    OnShow := TabSheetOnShow;
  end;

  { create editor and load file }
  FEditor := TSynEdit.Create(FTabSheet);
  with FEditor do
  begin
    Parent:=FTabsheet;
    Align:=alClient;
    Highlighter := FHighlighter;

    WantTabs:=True;
    Gutter.ShowLineNumbers := True;
    Gutter.DigitCount := 7;
    Gutter.Width := 42;

    BookMarkOptions.BookmarkImages := FImages;

    PopUpMenu := FPopUpMenu;

    SetDefaults (
          inAutoIndent,
          inInsertMode,
          inUseTabCharacter,
          inSmartTabs,
          inCursorBeyondEOF,
          inCursorBeyondEOL,
          inShowScrollHint,
          inTrimTrailingSpaces,
          inShowLineNumbers,
          inTabStops,
          inEditorFont,
          inFontSize );

    if ( FileExists ( FFileName ) ) then
      Lines.LoadFromFile( FFileName );

    OnReplaceText := FOnReplaceTextEvent;
    OnGutterClick := GutterClick;

    Modified := false;

    ReadOnly := inReadOnly;
  end;


  FDebuggerSupport := TDebugSupportPlugin.Create( FEditor );
  FDebuggerSupport.DebuggerList := FDebuggerList;
  FDebuggerSupport.Images := FImages;
  FDebuggerSupport.FileName := Filename;
  
  { link the debugger to this editor }
  FEditor.OnSpecialLineColors := SpecialLineColors;

  { switch to this sheet }
  FPageControl.ActivePage:=FTabSheet;

  editor.SetFocus;
end;

{ sets the focus to the editor when tabsheet is visible }
procedure TEditorObject.TabSheetOnShow ( Sender : TObject );
begin
    editor.SetFocus;
end;

destructor TEditorObject.Destroy;
begin
  DebuggerSupport.Free;
  FEditor.Free;
  FTabSheet.Free;

  inherited Destroy;
end;

procedure TEditorObject.SetFilename ( value : string );
begin
  FFilename := value;
  FDebuggerSupport.FileName := value;

  { change title on top of tabsheet}
  FTabSheet.caption := ExtractFileName(FFilename);;
end;

function TEditorObject.GetFileName : string;
begin
  { if the value is empty the we return untiled.cs }
  if ( FFilename = '' ) then
    Result := 'untitled.cs'
  else
    Result := FFilename;
end;

{ saves file with the filename is has }
function TEditorObject.Save( SaveEvenIfNotModified : boolean ) : boolean;
var
  Attrs : integer;
begin
  Result := False;
  { is this a new file? }
  if ( FFilename = '' ) then
    if ( Fsavedialog_save_file.execute ) then
      Filename := Fsavedialog_save_file.filename;

  { did they click cancel ? }
  if ( FFilename <> '' ) then
  begin
    { no, so has it been modified? }
    if ( Feditor.modified ) or ( SaveEvenIfNotModified ) then
    begin
      { one already there ? }
      if ( FileExists( FFilename ) ) then
      begin
        { then get current attributes }
        Attrs := FileGetAttr(FFilename);
        { read only ? }
        if ( ( Attrs and faReadOnly ) = faReadOnly ) then
        begin
          if ( MessageDlg(Format('%s has the read only flag set. To save this file'+#13+
          'this flag needs to be cleared. Do you want Tribal IDE to remove the flag and'+
          'save the file?' ,[FFilename]), mtConfirmation, [mbYes, mbNo], 0) = mrNo ) then
          begin
            { guess they dont trust us }
            Result := False;
            exit;
          end;

          { clear read only flag }
          Attrs := Attrs and not ( faReadOnly );
          { set the new attribute }
          FileSetAttr(FFilename, Attrs);
        end;
      end;

      { yes so save it }
      FEditor.Lines.SaveToFile ( FFilename );
      { now not different from file }
      Feditor.modified := False;

      if ( Assigned(FOnFileSave) ) then
        FOnFileSave( Self, FFilename );
    end;
    { save successful }
    Result := True;
  end;
end;

{ saves file with a new filename which it requests }
function TEditorObject.SaveRequest : boolean;
begin
  Result := False;
  if ( Fsavedialog_save_file.execute ) then
  begin
    SaveAs( Fsavedialog_save_file.filename );

    Result := True;
  end;
end;

{ saves file with a new filename }
procedure TEditorObject.SaveAs ( inFilename : string );
begin
  { save this filename }
  Filename := inFilename;
  { save the file }
  Save ( True );
end;



procedure TEditorObject.GotoLine( line_number : integer );
begin
 { goto correct line number }
  with FEditor do
  begin
    TopLine:=line_number-(LinesInWindow div 2);
    CaretY := line_number;
  end;
end;

procedure TEditorObject.FindGotoLine( line_number : integer );
begin
  GotoLine( line_number );

  FoundLine := line_number;        
end;

procedure TEditorObject.GutterClick(Sender: TObject; X, Y, Line: Integer; mark: TSynEditMark);
begin
  FDebuggerList.ToggleBreakPoint ( filename, Line );
end;


procedure TEditorObject.SpecialLineColors(Sender: TObject;
  Line: Integer; var Special: Boolean; var FG, BG: TColor);
var
  breakpoint_index : integer;
begin
  if ( FoundLine = Line ) then
  begin
      Special := TRUE;
      FG := clWhite;
      BG := clMaroon
  end;

  breakpoint_index := FDebuggerList.FindBreakPoint ( filename, line );
  if ( breakpoint_index <> -1 ) then
  begin
    if ( FDebuggerList.SteppingAtFile = filename ) and
       ( FDebuggerList.SteppingAtLine = Line ) then
    begin
      Special := TRUE;
      FG := clWhite;
      BG := clBlue;
    end
    else
      if ( FDebuggerList[breakpoint_index].Active ) and
         ( FDebuggerList[breakpoint_index].Enabled ) then
    begin
      Special := TRUE;
      FG := clWhite;
      BG := clRed
    end
    else if ( FDebuggerList[breakpoint_index].Active ) then
    begin
      Special := TRUE;
      FG := clRed;
      BG := clLime;
    end;
  end;
end;

procedure TEditorObject.SetFoundLine( value : integer );
begin
  FFoundLine := value;
  FLastX := FEditor.CaretX;
  FLastY := FEditor.CaretY;

  FEditor.invalidate;
end;

procedure TEditorObject.SetDefaults (
          inAutoIndent,
          inInsertMode,
          inUseTabCharacter,
          inSmartTabs,
          inCursorBeyondEOF,
          inCursorBeyondEOL,
          inShowScrollHint,
          inTrimTrailingSpaces,
          inShowLineNumbers : boolean;
          inTabStops : integer;
          inEditorFont : string;
          inFontSize : integer );
begin
  with FEditor do
  begin
    InsertMode := inInsertMode;
    gutter.ShowLineNumbers:=inShowLineNumbers;

    SetOptionFlag(eoAutoIndent, inAutoIndent);
    SetOptionFlag(eoTabsToSpaces, not inUseTabCharacter);
    SetOptionFlag(eoSmartTabs, inSmartTabs);
    SetOptionFlag(eoScrollPastEof, inCursorBeyondEOF);
    SetOptionFlag(eoScrollPastEoL, inCursorBeyondEOL);
    SetOptionFlag(eoShowScrollHint, inShowScrollHint);
    SetOptionFlag(eoTrimTrailingSpaces, inTrimTrailingSpaces);

    Font.Name := inEditorFont;
    Font.Size := inFontSize;

    TabWidth := inTabStops;
  end;
end;


procedure TDebugSupportPlugin.LinesInserted(FirstLine, Count: integer);
begin
  FDebuggerList.Inserted(FileName, FirstLine, Count);

  Editor.Invalidate;
end;

procedure TDebugSupportPlugin.LinesDeleted(FirstLine, Count: integer);
begin
  FDebuggerList.Deleted(FileName, FirstLine, Count);

  Editor.Invalidate;
end;

procedure TDebugSupportPlugin.AfterPaint(ACanvas: TCanvas; AClip: TRect;
  FirstLine, LastLine: integer);
var
  LH, X, Y: integer;
  ImgIndex: integer;
  breakpoint_Index : integer;
begin
  LH := Editor.LineHeight;
  Y := (LH - Images.Height) div 2
    + LH * (FirstLine - Editor.TopLine);
  while FirstLine <= LastLine do
  begin
    breakpoint_index := FDebuggerList.FindBreakPoint(Filename, FirstLine);

    if ( FDebuggerList.SteppingAtLine = FirstLine ) and
       ( FDebuggerList.SteppingAtFile = Filename )then
    begin
      X := 8 + 16;

      FImages.Draw(ACanvas, X, Y, 2);
    end;

    if ( breakpoint_index <> -1 ) then
    begin
      X := 8;

      if ( FDebuggerList[breakpoint_index].Active ) and
         ( FDebuggerList[breakpoint_index].Enabled ) then
        ImgIndex := 1
      else if ( FDebuggerList[breakpoint_index].Active ) then
        ImgIndex := 3
      else  
        ImgIndex := 0;


        FImages.Draw(ACanvas, X, Y, ImgIndex);
    end;

    Inc(FirstLine);
    Inc(Y, LH);
  end;
end;



end.
