unit unit_debugger_list;

interface

uses
  graphics, classes, sysutils, controls, comctrls, synedit, Eltree;

type


TSetBreakPoint = procedure ( Sender : TObject; filename : string; line_number : integer; breakpoint_status : boolean ) of object;

TDebuggerObject = class(TObject)
private
  FBreakPointLine  : integer;
  FActive          : boolean; { whether it a real break point ie. red line }
  FEnabled         : boolean; { whether it is currently enabled }
  FEditor          : TSynEdit;
  FFileName        : string;
  FCondition       : string;
  FPassCount       : integer;
  FGroup           : string;
  FElTree          : TElTree;
  FELTreeItem      : TELTreeItem;
  FBasePath        : string;
  FDisableAfterHit : boolean;
  FOnSetBreakPoint : TSetBreakPoint;


  procedure SetEditor ( value : TSynEdit );
  procedure SetActive ( value : boolean );
  procedure SetEnabled ( value : boolean );
  procedure SetPassCount ( value : integer );
  procedure SetGroup ( value : string );
  procedure SetCondition ( value : string );
  procedure SetFileName ( value : string );
  procedure SetBreakPointLine ( value : integer );
  procedure SetDisableAfterHit ( value : boolean );
public
  { -------------------------------------------------------------------------- }
  constructor Create(
              inBasePath : string;
              inFilename : string;
              inBreakPointLine : integer;
              inActive : boolean;
              inCondition : string;
              inPassCount : integer;
              inGroup : string;
              inElTree : TElTree;
              inOnSetBreakpoint : TSetBreakPoint );

  destructor Destroy; override;
  { -------------------------------------------------------------------------- }

  { -------------------------------------------------------------------------- }
  property OnSetBreakPoint : TSetBreakPoint read FOnSetBreakPoint write FOnSetBreakPoint;
  property BasePath : string read FBasePath write FBasePath;
  property ElTree : TElTree read FEltree write FEltree;
  property ElTreeItem : TElTreeItem read FElTreeItem write FElTreeItem;
  property Condition : string read FCondition write SetCondition;
  property PassCount : integer read FPassCount write SetPassCount;
  property Group : string read FGroup write SetGroup;
  property DisableAfterHit : boolean read FDisableAfterHit write SetDisableAfterHit;
  property Editor : TSynedit read FEditor write SetEditor;
  property FileName : string read FFilename write SetFilename;
  property BreakPointLine : integer read FBreakPointLine write SetBreakPointLine;
  property Active : boolean read FActive write SetActive;
  property Enabled : boolean read FEnabled write SetEnabled;
  { -------------------------------------------------------------------------- }
end;

TDebuggerList = class(TList)
private
  { -------------------------------------------------------------------------- }
  FSteppingAtLine       : integer;
  FSteppingAtFile       : string;
  FOnSetBreakPoint      : TSetBreakPoint;
  FEltree               : TElTree;
  FBasePath             : string;
  FEnabled              : boolean;
  { -------------------------------------------------------------------------- }

  procedure SetSteppingAtLine ( value : integer );
protected
  { -------------------------------------------------------------------------- }
  function GetDebuggerObject(Index: Integer): TDebuggerObject;
  procedure SetDebuggerObject(Index: Integer; Value: TDebuggerObject);
  procedure SetCount(Value: Integer);
  { -------------------------------------------------------------------------- }
public
  { -------------------------------------------------------------------------- }
  constructor Create; overload;
  destructor Destroy; override;
  procedure Clear; override;
  procedure Delete(index: Integer);
  procedure DeleteFilesBreakpoints(filename : string);
  procedure DeleteFilesNonSetBreakpoints(filename : string);
  function FindBreakPoint( filename : string; line_number : integer ) : integer;
  procedure ToggleBreakPoint ( filename : string; line_number : integer );
  function AddBreakPoint (
            filename : string;
            line_number : integer;
            Condition : string;
            PassCount : integer;
            Group : string ) : integer;
  procedure Deleted( filename : string; start_line, line_count : integer );
  procedure Inserted( filename : string; start_line, line_count : integer );
  procedure ResendBreakPoints ( filename : string; invert : boolean ) ;
  procedure SetBreakPoints( filename, breakpoints : string );
  procedure SetEditor ( filename : string; Editor : TSynEdit );
  procedure GetGroups ( var GroupList : TStringList );
  procedure ChangeGroupEnableState ( group_name : string; new_state : boolean );
  { -------------------------------------------------------------------------- }

  { -------------------------------------------------------------------------- }
  property Enabled : boolean read FEnabled write FEnabled;
  property BasePath : string read FBasePath write FBasePath;
  property ElTree : TElTree read FEltree write FEltree;
  property OnSetBreakPoint : TSetBreakPoint read FOnSetBreakPoint write FOnSetBreakPoint;
  property SteppingAtLine : integer read FSteppingAtLine write SetSteppingAtLine;
  property SteppingAtFile : string read FSteppingAtFile write FSteppingAtFile;
  property breakpoints[Index: Integer]: TDebuggerObject read GetDebuggerObject write SetDebuggerObject; default;
  property Count write SetCount;
  { -------------------------------------------------------------------------- }
end;

implementation

uses unit_editor_utils;

{ create object and initialise image indexes }
constructor TDebuggerList.Create;
begin
  inherited Create;

  FSteppingAtLine := -1;
  FEnabled := False;
end;

{ Clear all the DebuggerObjects from the list and destroy the list. }
destructor TDebuggerList.Destroy;
begin
  Clear;
  inherited Destroy;
end;

{ Return an DebuggerObject from the list. }
function TDebuggerList.GetDebuggerObject(Index: Integer): TDebuggerObject;
begin
  Result := TDebuggerObject(Items[Index]);
end;

{ Set an DebuggerObject in the list. Free the old DebuggerObject. }
procedure TDebuggerList.SetDebuggerObject(index: Integer; value: TDebuggerObject);
begin
  breakpoints[index].free;
  items[index] := pointer(value);
end;

{ Clear the list by deleting all DebuggerObjects in it. }
procedure TDebuggerList.Clear;
var
  i : Integer;
begin

  with FElTree do
  begin
    HeaderSections.Clear;

    with HeaderSections.AddSection do
    begin
      Text := 'Filename';
      AutoSize:=True;
    end;

    with HeaderSections.AddSection do
    begin
      Text := 'Line Number';
      AutoSize:=True;
    end;

    with HeaderSections.AddSection do
    begin
      Text := 'Condition';
      AutoSize:=True;
    end;

    with HeaderSections.AddSection do
    begin
      Text := 'Disable After';
      AutoSize:=True;
    end;

    with HeaderSections.AddSection do
    begin
      Text := 'Pass Count';
      AutoSize:=True;
    end;

    with HeaderSections.AddSection do
    begin
      Text := 'Group';
      AutoSize:=True;
    end;

    { make headers visible }
    ShowColumns:=True;
  end;


  { delete all the breakpoint objects }
  for i := count-1 downto 0 do
    Delete ( i );

  { should be clear already, but call inheritied clear anyway }
  inherited clear;
end;

{ Delete an DebuggerObject from the list, freeing the DebuggerObject }
procedure TDebuggerList.Delete(index: Integer);
begin
  breakpoints[index].Free;

  inherited delete(index);
end;

{ If the list shrinks, free the DebuggerObjects that are implicitly deleted. }
procedure TDebuggerList.SetCount(value: Integer);
begin
  while value < count do
    delete(count-1);

  inherited count := value;
end;

{ find the breakpoint at the specified line number }
function TDebuggerList.FindBreakPoint( filename : string; line_number : integer ) : integer;
var
  i : integer;
begin
  Result := -1;

  for i:=0 to Count-1 do
  begin
    { found it ? }
    if ( breakpoints[i].BreakPointLine = line_number ) and
       ( breakpoints[i].Filename = filename ) then
    begin
      { return index }
      Result := i;
      exit;
    end;
  end;
end;

{ toggle the value currently at the specified line number }
procedure TDebuggerList.ToggleBreakPoint ( filename : string; line_number : integer );
var
  breakpoint_index : integer;
begin
  if ( not Enabled ) then
    exit;

  { find it }
  breakpoint_index := FindBreakPoint ( filename, line_number );
  { found ? }
  if ( breakpoint_index <> -1 ) then
  begin
    { toggle it }
    breakpoints[breakpoint_index].Active := not breakpoints[breakpoint_index].Active;
  end
end;


{ Trigger the breakpoint event  }
procedure TDebuggerList.ResendBreakPoints ( filename : string; invert : boolean ) ;
var
  i : integer;

begin
  for i := 0 to Count-1 do
  begin
    if ( Assigned ( FOnSetBreakPoint ) ) and
       ( breakpoints[i].Active ) and
       ( breakpoints[i].Filename = filename ) then
    begin
      { breakpoint enabled ? }
      if ( breakpoints[i].Enabled ) then
      begin
        { set/clear breakpoint }
        with breakpoints[i] do
          FOnSetBreakPoint ( breakpoints[i], Filename, BreakPointLine, not invert );
      end
      else
      begin
        { if disabled we always send the clr breakpoint command }
        with breakpoints[i] do
          FOnSetBreakPoint ( breakpoints[i], Filename, BreakPointLine, false );
      end;
    end;
  end
end;

{ add a new breakpoint to the list at specified line number }
function TDebuggerList.AddBreakPoint (
                        filename : string;
                        line_number : integer;
                        Condition : string;
                        PassCount : integer;
                        Group : string ) : integer;
var
  temp_breakpoint  : TDebuggerObject;
begin
  Result := FindBreakPoint ( filename, line_number );

  if ( Result = -1 ) then
  begin
    temp_breakpoint := TDebuggerObject.Create(FBasePath, filename, line_number, False,
                                              Condition, PassCount, Group, FElTree, FOnSetBreakPoint);
    Add ( temp_breakpoint );

    Result := Count-1;
  end;
end;


procedure TDebuggerList.SetSteppingAtLine (  value : integer );
begin
  FSteppingAtLine := Value;
end;

procedure TDebuggerList.Deleted( filename : string; start_line, line_count : integer );
var
  i : integer;
begin
  { alter our stepping at line to match with deleted lines }
  if ( FSteppingAtLine >= start_line + line_count ) then
    FSteppingAtLine:=FSteppingAtLine-line_count
  else
  if ( FSteppingAtLine > start_line ) then
    FSteppingAtLine := start_line;

  { now adjust the breakpoints }
  for i:=Count -1 downto 0 do
  begin
    if ( breakpoints[i].BreakPointLine >= start_line + line_count ) and
       ( filename = breakpoints[i].filename ) then
      breakpoints[i].BreakPointLine:=breakpoints[i].BreakPointLine-line_count
    else
    if ( breakpoints[i].BreakPointLine > start_line ) then
      Delete(i);
  end;
end;

procedure TDebuggerList.Inserted( filename : string; start_line,  line_count : integer );
var
  i : integer;
begin
  { change our stepping at line }
  if ( FSteppingAtLine >= start_line ) then
    FSteppingAtLine:=FSteppingAtLine+line_count;

  { there have been lines inserted change our breakpoints to match }
  for i:=Count-1 downto 0 do
  begin
    if ( breakpoints[i].BreakPointLine >= start_line ) and
       ( filename = breakpoints[i].filename ) then
        breakpoints[i].BreakPointLine:=breakpoints[i].BreakPointLine+line_count;
  end;
end;

procedure TDebuggerList.DeleteFilesBreakpoints(filename : string);
var
  i : integer;
begin
  { delete all the breakpoint objects }
  for i := count-1 downto 0 do
    if ( breakpoints[i].FileName = filename ) then
      Delete ( i );
end;

procedure TDebuggerList.DeleteFilesNonSetBreakpoints(filename : string);
var
  i : integer;
begin
  { delete all the breakpoint objects }
  for i := count-1 downto 0 do
    if ( breakpoints[i].FileName = filename ) and
       ( not breakpoints[i].Active ) then
      Delete ( i );
end;

{ sets the breakpoints for a file }
procedure TDebuggerList.SetBreakPoints( filename, breakpoints : string );
var
  BreakpointCount : integer;
  i,j             : integer;
  line_skip       : integer;
  number_of_breaks: integer;
  line_position   : integer;
begin
  BreakpointCount := StrToInt(PacketParamStr(' ',0, breakpoints));

  line_position := 1;

  for i:=0 to BreakpointCount-1 do
  begin
    line_skip:=StrToInt(PacketParamStr(' ',(i*2)+1, breakpoints));
    number_of_breaks:=StrToInt(PacketParamStr(' ',(i*2)+2, breakpoints));
    inc ( line_position, line_skip );


    for j:=0 to number_of_breaks - 1 do
    begin
      AddBreakPoint( filename, line_position, '', 0, '');

      inc ( line_position );
    end;

  end;
end;

procedure TDebuggerList.SetEditor ( filename : string; Editor : TSynEdit );
var
  i : integer;

begin
  { set all the breakpoints with the filename = filename this editor }
  for i := Count-1 downto 0 do
  begin
    if ( lowercase(breakpoints[i].filename) = lowercase(filename) ) then
    begin
      breakpoints[i].Editor := Editor;
    end;
  end;
end;

procedure TDebuggerList.GetGroups ( var GroupList : TStringList );
var
  i : integer;
begin
  GroupList.Clear;

  { go through active breakpoints and pick ones that dont have '' for group }
  for i:=0 to Count-1 do
  begin
    if ( breakpoints[i].Active ) and
       ( breakpoints[i].Group <> '' ) and
       ( grouplist.indexof(lowercase(breakpoints[i].Group)) = -1 ) then
    begin
      GroupList.Add( lowercase(breakpoints[i].Group) );
    end;
  end;
end;

procedure TDebuggerList.ChangeGroupEnableState ( group_name : string; new_state : boolean );
var
  i : integer;
begin
  for i := 0 to Count-1 do
  begin
    if ( lowercase(breakpoints[i].Group) = lowercase(group_name) ) then
    begin
      breakpoints[i].Enabled := new_state;
    end;
  end;
end;

{ ============================================================================ }

{ create Debugger object and set filename }
constructor TDebuggerObject.Create(
            inBasePath : string;
            inFilename : string;
            inBreakPointLine : integer;
            inActive : boolean;
            inCondition : string;
            inPassCount : integer;
            inGroup : string;
            inElTree : TElTree;
            inOnSetBreakpoint : TSetBreakPoint );
begin
  inherited Create;

  FOnSetBreakpoint := inOnSetBreakPoint;

  FBasePath := inBasePath;
  FFilename := inFilename;
  FCondition := inCondition;
  FPassCount := inPassCount;
  FGroup := inGroup;
  FEnabled := True;
  FBreakPointLine := inBreakpointLine;
  FElTree := inELtree;

  { causes event }
  Active := inActive;
end;

{ free Debugger object }
destructor TDebuggerObject.Destroy;
begin
  if ( Assigned(FELTreeItem) ) and ( Assigned(FElTree) ) then
  begin
    FElTree.items.DeleteItem(FElTreeItem);
  end;

  FFilename := '';

  { tell it to redraw so it can see our break point }
  if ( Assigned(FEditor) ) then
    FEditor.InvalidateLine(FBreakPointLine);

  inherited Destroy;
end;

procedure TDebuggerObject.SetActive ( value : boolean );
begin
  if ( FActive <> value ) then
  begin
    FActive := Value;
    FEnabled := True;

    { breakpoint being removed reset condition/passcount/etc }
    if ( not FActive ) then
    begin
      FCondition := '';
      FPassCount := 0;
      FEnabled := True;
      FGroup := '';
      FDisableAfterHit := False;
    end;

    { generate set break point event }
    if ( Assigned ( FOnSetBreakPoint ) ) then
        FOnSetBreakPoint ( Self, FFilename, FBreakPointLine, FActive );
  end;

  if ( Assigned(FElTree) ) then
  begin
    FElTree.Items.BeginUpdate;

    try
      if ( not FActive ) and ( Assigned(FELTreeItem) ) then
      begin
        FElTree.items.DeleteItem( FELTreeItem );

        FELTreeItem := nil;
      end
      else if ( FActive ) then
      begin
        if ( not Assigned(FELTreeItem) ) then
        begin
          { create eltree item }
          FELTreeItem := FElTree.items.AddObject( nil, '', Self);
        end;

        FELTreeItem.Text := ExtractRelativePath(FBasePath, FFilename)+'  ';
        with FELTreeItem.ColumnText do
        begin
          Clear;
          
          Add(IntToStr(FBreakPointLine));
          Add(FCondition);
          if ( FDisableAfterHit ) then
            Add('True')
          else
            Add('False');
          Add(IntToStr(FPassCount));
          Add(FGroup);
        end;
      end;

      if ( Assigned(FELTreeItem) ) then
      begin
        if ( FEnabled ) then
          FELTreeItem.ImageIndex := 37
        else
          FELTreeItem.ImageIndex := 38;
      end;
    finally
      FElTree.Items.EndUpdate;
    end;
  end;

  { redraw editor }
  if ( Assigned(FEditor) ) then
    FEditor.InvalidateLine(FBreakPointLine)
end;


procedure TDebuggerObject.SetEnabled ( value : boolean );
begin
  FEnabled := Value;

  { if we have been renabled then we re-active breakpoint,
  otherwise we clear it }
  if ( Assigned ( FOnSetBreakPoint ) ) then
      FOnSetBreakPoint ( Self, FFilename, FBreakPointLine, FEnabled);

  if ( Assigned(FELTreeItem) ) then
  begin
    if ( FEnabled ) then
      FELTreeItem.ImageIndex := 37
    else
      FELTreeItem.ImageIndex := 38;
  end;
  
  { tell it to redraw so it can change }
  if ( Assigned(FEditor) ) then
    FEditor.InvalidateLine(FBreakPointLine)
end;


procedure TDebuggerObject.SetEditor ( value : TSynEdit );
begin
  FEditor := value;

  { tell it to redraw so it can see our break point }
  if ( Assigned(FEditor) ) then
    FEditor.InvalidateLine(FBreakPointLine)
end;

procedure TDebuggerObject.SetBreakPointLine ( value : integer );
begin
  { changed ? }
  if ( Value <> FBreakPointLine ) then
  begin
    { clear what its currently set to }
    if ( Assigned ( FOnSetBreakPoint ) ) then
        FOnSetBreakPoint ( Self, FFilename, FBreakPointLine, False );

    if ( Assigned(FEditor) ) then
      FEditor.InvalidateLine(FBreakPointLine);

    FBreakPointLine := value;
  end;
  
  { and set to this new line }
  if ( Assigned ( FOnSetBreakPoint ) ) then
    FOnSetBreakPoint ( Self, FFilename, FBreakPointLine, FActive );

  { tell it to redraw so it can see our break point }
  if ( Assigned(FEditor) ) then
    FEditor.InvalidateLine(FBreakPointLine)
end;

procedure TDebuggerObject.SetPassCount ( value : integer );
begin
  { changed ? }
  if ( value <> FPassCount ) then
  begin
    { clear previous pass count }
    if ( Assigned ( FOnSetBreakPoint ) ) then
        FOnSetBreakPoint ( Self, FFilename, FBreakPointLine, False );

    FPassCount := value;

    { and set to this one }
    if ( Assigned ( FOnSetBreakPoint ) ) then
        FOnSetBreakPoint ( Self, FFilename, FBreakPointLine, FActive );
  end;

  { tell it to redraw so it can see our break point }
  if ( Assigned(FEditor) ) then
    FEditor.InvalidateLine(FBreakPointLine)
end;

procedure TDebuggerObject.SetGroup ( value : string );
begin
  FGroup := value;

  { tell it to redraw so it can see our break point }
  if ( Assigned(FEditor) ) then
    FEditor.InvalidateLine(FBreakPointLine)
end;

procedure TDebuggerObject.SetCondition ( value : string );
begin
  { changed ? }
  if ( value <> FCondition) then
  begin
    { clear previous Condition }
    if ( Assigned ( FOnSetBreakPoint ) ) then
        FOnSetBreakPoint ( Self, FFilename, FBreakPointLine, False );

    FCondition := value;

    { and set to this one }
    if ( Assigned ( FOnSetBreakPoint ) ) then
        FOnSetBreakPoint ( Self, FFilename, FBreakPointLine, FActive );
  end;

  { tell it to redraw so it can see our break point }
  if ( Assigned(FEditor) ) then
    FEditor.InvalidateLine(FBreakPointLine)
end;

procedure TDebuggerObject.SetDisableAfterHit ( value : boolean );
begin
  { changed ? }
  if ( value <> FDisableAfterHit ) then
  begin
    { clear previous Condition }
    if ( Assigned ( FOnSetBreakPoint ) ) then
        FOnSetBreakPoint ( Self, FFilename, FBreakPointLine, False );

    FDisableAfterHit := value;

    { and set to this one }
    if ( Assigned ( FOnSetBreakPoint ) ) then
        FOnSetBreakPoint ( Self, FFilename, FBreakPointLine, FActive );
  end;

  { tell it to redraw so it can see our break point }
  if ( Assigned(FEditor) ) then
    FEditor.InvalidateLine(FBreakPointLine)
end;

procedure TDebuggerObject.SetFileName ( value : string );
begin
  { changed ? }
  if ( Value <> FFilename ) then
  begin
    { clear previous FFilename }
    if ( Assigned ( FOnSetBreakPoint ) ) then
        FOnSetBreakPoint ( Self, FFilename, FBreakPointLine, False );

    FFilename := value;

    { and set to this one }
    if ( Assigned ( FOnSetBreakPoint ) ) then
        FOnSetBreakPoint ( Self, FFilename, FBreakPointLine, FActive );
  end;


  { tell it to redraw so it can see our break point }
  if ( Assigned(FEditor) ) then
    FEditor.InvalidateLine(FBreakPointLine)
end;




end.
